/**
 * 
 */
package com.ejie.ab18a.control;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import com.ejie.ab18a.entidades.Ab18aUsuario;
import com.ejie.ab18a.model.Ab18aEnvio;
import com.ejie.ab18a.model.Ab18aFiltroMonitor;
import com.ejie.ab18a.model.Ab18aPeticionDetalle;
import com.ejie.ab18a.model.Ab18aPeticionEnvio;
import com.ejie.ab18a.service.Ab18aPeticionEnvioService;
import com.ejie.ab18a.utils.Ab18aAccesoWS;
import com.ejie.ab18a.utils.Ab18aConvertirJaxb;
import com.ejie.ab18a.utils.Constantes;
import com.ejie.ab18a.utils.Utilidades;
import com.ejie.ab18a.ws.model.Ab18aDatosRespuestaEsentool;
import com.ejie.ab18a.ws.model.DatosBoletin;
import com.ejie.x38.dto.JQGridRequestDto;
import com.ejie.x38.dto.JQGridResponseDto;
import com.ejie.x38.json.JSONObject;
import com.ejie.x38.security.UserCredentials;
import com.ejie.xmlbeans.ServicioObtencionPerfilesUsuarios;

import eu.europa.ec.publications.esentool.rest.client.EsentoolRestClient;
import eu.europa.ec.publications.esentool.rest.client.EsentoolRestClient.EsentoolRestClientBuilder;
import eu.europa.ec.publications.esentool.rest.client.RenderFormat;
import eu.europa.ec.publications.esentool.rest.client.RestClientException;

/**
 * @author XXXX
 * 
 */
@Controller
@RequestMapping(value = "monitor")
public class Ab18aMonitorController {

	/** The Constant logger. */
	private static final Logger logger = LoggerFactory.getLogger(Ab18aMonitorController.class);

	@Autowired
	private Ab18aPeticionEnvioService ab18aPeticionEnvioService;

	@Autowired
	private ReloadableResourceBundleMessageSource messageSource;

	// /** The message source. */
	// @Resource()
	// private ReloadableResourceBundleMessageSource messageSource;

	/** Obtiene la JSP del monitor **/
	@RequestMapping(value = "viewMonitor", method = RequestMethod.GET)
	public String viewMonitor() {
		logger.info("Entramos en el monitor de envíos");
		return "ab18aMonitor";
	}

	@RequestMapping(value = "/peticion/detalle", method = RequestMethod.GET)
	public @ResponseBody
	Ab18aPeticionDetalle viewDetalle(@RequestParam(value = "idPeticion") Integer idPeticion) {
		logger.info("Entramos en el detalle de una petición");
		Ab18aPeticionDetalle detallePeticion = new Ab18aPeticionDetalle();

		try {
			detallePeticion = this.ab18aPeticionEnvioService.findDetallePeticionById(idPeticion);
			detallePeticion = parsearFechas(detallePeticion);
		} catch (Exception e) {
			logger.error("Error al cargar el detalle");
		}

		return detallePeticion;
	}

	/**
	 * Method 'getCreateForm'. Llamado para cargar el detalle de un expediente
	 * en modificacion
	 * 
	 * @param idExpediente
	 *            Long
	 * @param model
	 *            Model
	 * @return ModelAndView
	 */
	@RequestMapping(value = "/peticion", method = RequestMethod.GET)
	public ModelAndView getCreateFormExpediente(@RequestParam(value = "idPeticion") Integer idPeticion, Model model) {
		Ab18aMonitorController.logger.error("[GET - View] : ab18aPeticionDetalle");
		model.addAttribute("idPeticionEnvio", idPeticion);
		return new ModelAndView("ab18aPeticionDetalle", "model", model);
	}

	/** Buscamos las peticiones correspondientes a los valores del filtro **/
	@RequestMapping(value = "peticiones/tabla", method = RequestMethod.GET, headers = { "JQGridModel=true" })
	public @ResponseBody
	JQGridResponseDto<Ab18aEnvio> buscarPeticiones(@ModelAttribute() Ab18aFiltroMonitor filtroMonitor, @ModelAttribute() JQGridRequestDto request,
			HttpServletRequest httpRequest) {

		HttpSession session = httpRequest.getSession();
		Ab18aUsuario usuario = null;
		if (session.getAttribute("usuario") != null) {
			usuario = (Ab18aUsuario) session.getAttribute("usuario");
		} else {
			UserCredentials userCredentials = null;
			userCredentials = (UserCredentials) SecurityContextHolder.getContext().getAuthentication().getCredentials();
			String dniUsuarioLogeado = Utilidades.calculoLetraDNI(userCredentials.getNif());
			ServicioObtencionPerfilesUsuarios perfil = null;
			usuario = new Ab18aUsuario();
			try {
				String[] params = { "dni" };
				String[] values = { dniUsuarioLogeado };

				String resultado = Ab18aAccesoWS.obtenerInformacionWS("obtenerPerfilesUsuariosOSBUS", params, values);

				Ab18aConvertirJaxb conversion = new Ab18aConvertirJaxb();

				if (resultado != null) {
					perfil = conversion.xmlToPerfiles(resultado);
				}

				if (perfil != null && perfil.getIdUsuario() != 0) {

					usuario.setDni(dniUsuarioLogeado);
					Utilidades.rellenarUsuario(usuario, perfil);

					session.setAttribute("usuario", usuario);
				}
			} catch (Exception e) {
				logger.error("Error al obtener el usuario getInformacionUsuario:\t" + e.getMessage());
			}
		}

		List<Ab18aEnvio> listaEnvios = new ArrayList<Ab18aEnvio>();
		Long recordNum = null;

		try {
			listaEnvios = this.ab18aPeticionEnvioService.findPeticionesCuadro(filtroMonitor, usuario, request);
			recordNum = this.ab18aPeticionEnvioService.findAllCount(filtroMonitor, usuario);
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - buscarPeticiones() - Error al obtener las peticiones - " + e);
		}

		JQGridResponseDto<Ab18aEnvio> response = new JQGridResponseDto<Ab18aEnvio>(request, recordNum, listaEnvios);

		return response;
	}

	@RequestMapping(value = "reactivarPeticion", method = RequestMethod.POST)
	public @ResponseBody
	void reactivarPeticion(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {
		try {
			if (Utilidades.notNullAndNotZero(idPeticion)) {
				Integer numIntentos = this.ab18aPeticionEnvioService.findNumIntentosById(idPeticion);
				if (numIntentos == null || numIntentos < 5) {
					response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
							this.messageSource.getMessage("cuadro.validacion.reactivarPeticion.numIntentos", null, LocaleContextHolder.getLocale()));
				} else {
					this.ab18aPeticionEnvioService.reactivar(idPeticion);
				}
			}
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - reactivarPeticion() - Error al reactivar la petición - " + e);
			JSONObject message = new JSONObject();
			message.append("mensajeError",
					this.messageSource.getMessage("cuadro.validacion.reactivarPeticion", null, LocaleContextHolder.getLocale()));
			try {
				response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED, message.toString());
			} catch (Exception e2) {
				Ab18aMonitorController.logger.error("Ab18aMonitorController - reactivarPeticion() - Error al reactivar la petición - " + e2);
			}
		}
	}

	@RequestMapping(value = "visualizarLink", method = RequestMethod.GET)
	public @ResponseBody
	List<String> visualizarLink(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {

		// String arrayToJson = "";

		List<String> listaLinks = new ArrayList<String>();

		try {
			listaLinks = this.ab18aPeticionEnvioService.findTedLinks(idPeticion);

		} catch (Exception e) {
			Ab18aMonitorController.logger
					.error("Ab18aMonitorController - visualizarLink() - Error al recuperar la lista de links de la publicación - " + e);
			try {
				response.sendError(400, "No se pudo recuperar la lista de links de la publicación  ");
			} catch (IOException e1) {
				Ab18aMonitorController.logger.error("Ab18aMonitorController - visualizarLink() - Error al devolver el callback del error - " + e1);
			}
		}

		if (Utilidades.nullOrEmpty(listaLinks)) {
			try {
				response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
						this.messageSource.getMessage("cuadro.validacion.linkPublicacionNoExiste", null, LocaleContextHolder.getLocale()) + "  ");
			} catch (IOException e1) {
				Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al devolver el callback del error - " + e1);
			}
		}
		// } else {
		// ObjectMapper objectMapper = new ObjectMapper();
		// // Set pretty printing of json
		// objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
		//
		// try {
		// arrayToJson = objectMapper.writeValueAsString(listaLinks);
		// } catch (JsonProcessingException e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
		// }

		return listaLinks;
	}

	@RequestMapping(value = "comprobarAnuncio", method = RequestMethod.GET)
	public @ResponseBody
	boolean comprobarAnuncio(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response)
			throws IOException {
		// Client eSentool
		EsentoolRestClient client = null;

		// Obtenemos los datos de la petición
		Ab18aDatosRespuestaEsentool datos = this.ab18aPeticionEnvioService.findRespuestaWS(idPeticion);

		// Instanciamos el cliente de eSentool
		try {
			client = new EsentoolRestClientBuilder(Constantes.URL_ERPI_SUBMIT_NOTICE, Constantes.ESENTOOL_USER, Constantes.ESENTOOL_PW).build();
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al instanciar el cliente - " + e);
		}

		if (client != null) {
			if (datos != null && Utilidades.notNullAndNotEmpty(datos.getIdEnvioBoletin())) {
				try {
					Ab18aMonitorController.logger.debug("Ab18aMonitorController - renderizarAnuncio() - Llamada a renderizar anuncio");
					byte[] anuncio = Base64.decodeBase64(client.renderNotice(datos.getIdEnvioBoletin(), RenderFormat.PDF, "ES"));
					Ab18aMonitorController.logger.debug("Ab18aMonitorController - renderizarAnuncio() - Anuncio renderizado");
					if (anuncio != null) {
						return true;
					}
				} catch (RestClientException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al renderizar el anuncio - " + e);
					try {
						response.sendError(
								Constantes.HTTP_STATUS_EXPECTATION_FAILED,
								this.messageSource.getMessage("cuadro.renderizar.errorRenderizar", null, LocaleContextHolder.getLocale())
										+ e.getMessage() + "  ");
						response.flushBuffer();
					} catch (IOException e1) {
						Ab18aMonitorController.logger
								.error("Ab18aMonitorController - renderizarAnuncio() - Error al devolver el callback del error - " + e1);
					}
				}
			} else {
				try {
					response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
							this.messageSource.getMessage("cuadro.renderizar.noExiste", null, LocaleContextHolder.getLocale()) + "  ");
					response.flushBuffer();
				} catch (IOException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al devolver el callback del error - "
							+ e);
				}
			}
		}
		return false;
	}

	@RequestMapping(value = "renderizarAnuncio", method = RequestMethod.GET)
	public @ResponseBody
	void renderizarAnuncio(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) throws IOException {

		// Client eSentool
		EsentoolRestClient client = null;

		// Obtenemos los datos de la petición
		Ab18aDatosRespuestaEsentool datos = this.ab18aPeticionEnvioService.findRespuestaWS(idPeticion);

		// Instanciamos el cliente de eSentool
		try {
			client = new EsentoolRestClientBuilder(Constantes.URL_ERPI_SUBMIT_NOTICE, Constantes.ESENTOOL_USER, Constantes.ESENTOOL_PW).build();
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al instanciar el cliente - " + e);
		}

		String nombre = "anuncio_render";
		String extension = "pdf";

		if (client != null) {
			if (datos != null && Utilidades.notNullAndNotEmpty(datos.getIdEnvioBoletin())) {
				try {
					Ab18aMonitorController.logger.debug("Ab18aMonitorController - renderizarAnuncio() - Llamada a renderizar anuncio");
					byte[] anuncio = Base64.decodeBase64(client.renderNotice(datos.getIdEnvioBoletin(), RenderFormat.PDF, "ES"));
					Ab18aMonitorController.logger.debug("Ab18aMonitorController - renderizarAnuncio() - Anuncio renderizado");
					if (anuncio != null) {
						response.setHeader("Pragma", "no-cache");
						response.setHeader("Expires", "-1");
						response.setHeader("Cache-Control", "no-store, no-cache, must-revalidate");
						response.setHeader("Content-disposition",
								"attachment;name=\"" + nombre + "\";filename=\"" + nombre + "." + extension.toLowerCase() + "\"");
						response.setContentType("application/octet-stream");
						response.getOutputStream().write(anuncio);
						response.getOutputStream().flush();
						response.getOutputStream().close();
						response.flushBuffer();
					}
				} catch (RestClientException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al renderizar el anuncio - " + e);
					try {
						response.sendError(
								Constantes.HTTP_STATUS_EXPECTATION_FAILED,
								this.messageSource.getMessage("cuadro.renderizar.errorRenderizar" + " " + e.getMessage(), null,
										LocaleContextHolder.getLocale())
										+ "  ");
						response.flushBuffer();
					} catch (IOException e1) {
						Ab18aMonitorController.logger
								.error("Ab18aMonitorController - renderizarAnuncio() - Error al devolver el callback del error - " + e1);
					}
				}
			} else {
				try {
					response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
							this.messageSource.getMessage("cuadro.renderizar.noExiste", null, LocaleContextHolder.getLocale()) + "  ");
					response.flushBuffer();
				} catch (IOException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - renderizarAnuncio() - Error al devolver el callback del error - "
							+ e);
				}
			}
		}

	}

	@RequestMapping(value = "paralizar", method = RequestMethod.GET)
	public @ResponseBody
	void paralizarPeticion(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {
		try {
			this.ab18aPeticionEnvioService.paralizarReanudarPeticion(idPeticion, Constantes.PETICION_PARALIZADA);
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - paralizarPeticion() - Error al paralizar la petición - " + e);
			try {
				response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
						this.messageSource.getMessage("cuadro.validacion.errorParalizar", null, LocaleContextHolder.getLocale()) + "  ");
			} catch (IOException io) {
				Ab18aMonitorController.logger.error("Ab18aMonitorController - paralizarPeticion() - Error al devolver el callback del error - " + e);
			}
		}
	}

	@RequestMapping(value = "reanudar", method = RequestMethod.GET)
	public @ResponseBody
	void reanudarPeticion(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {
		try {
			this.ab18aPeticionEnvioService.paralizarReanudarPeticion(idPeticion, Constantes.NUMERO_0);
		} catch (Exception e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - reanudarPeticion() - Error al reanudar la petición - " + e);
			try {
				response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
						this.messageSource.getMessage("cuadro.validacion.errorReanudar", null, LocaleContextHolder.getLocale()) + "  ");
			} catch (IOException io) {
				Ab18aMonitorController.logger.error("Ab18aMonitorController - reanudarPeticion() - Error al devolver el callback del error - " + e);
			}
		}
	}

	@RequestMapping(value = "descargarEnviado", method = RequestMethod.GET)
	public @ResponseBody
	String descargarEnviado(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {

		String xmlStr = "";

		DatosBoletin datos = this.ab18aPeticionEnvioService.findPeticionById(idPeticion);
		InputStream fileInputStream = null;
		try {

			if (Utilidades.notNullAndNotEmpty(datos) && Utilidades.notNullAndNotEmpty(datos.getXmlEnvioBoletin())) {

				byte[] xmlBytes = datos.getXmlEnvioBoletin().getBytes(Charset.forName("UTF-8"));
				fileInputStream = new ByteArrayInputStream(xmlBytes);

				fileInputStream.read(xmlBytes, 0, xmlBytes.length);

				if (Utilidades.notNullAndNotEmpty(fileInputStream)) {
					fileInputStream.close();
				}

				xmlStr = Base64.encodeBase64String(xmlBytes);
			} else {
				try {
					response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
							this.messageSource.getMessage("cuadro.validacion.xmlNoDisponible", null, LocaleContextHolder.getLocale()) + "  ");
				} catch (IOException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al devolver el callback del error - "
							+ e);
				}
			}

			if (Utilidades.notNullAndNotEmpty(fileInputStream)) {
				fileInputStream.close();
			}

		} catch (FileNotFoundException e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al obtener el XML - " + e);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al escribir el XML en el fichero - " + e);
		}

		return xmlStr;
	}

	@RequestMapping(value = "descargarZip", method = RequestMethod.GET)
	public @ResponseBody
	void descargarZIP(@RequestParam(value = "idPeticion", required = true) Integer idPeticion, HttpServletResponse response) {
		// String zipStr = "";

		Ab18aPeticionEnvio envio = new Ab18aPeticionEnvio();
		String nombre = "xmlZipBoletines";
		String extension = "zip";
		try {
			envio = this.ab18aPeticionEnvioService.findZipById(idPeticion);
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		// InputStream fileInputStream = null;
		try {

			if (Utilidades.notNullAndNotEmpty(envio) && Utilidades.notNullAndNotEmpty(envio.getT01AnuncioZip())) {
				byte[] xmlBytes = envio.getT01AnuncioZip();

				if (xmlBytes != null) {
					response.setHeader("Pragma", "no-cache");
					response.setHeader("Expires", "-1");
					response.setHeader("Cache-Control", "no-store, no-cache, must-revalidate");
					response.setHeader("Content-disposition",
							"attachment;name=\"" + nombre + "\";filename=\"" + nombre + "." + extension.toLowerCase() + "\"");
					response.setContentType("application/octet-stream");
					response.getOutputStream().write(xmlBytes);
					response.getOutputStream().flush();
					response.getOutputStream().close();
					response.flushBuffer();
				}
				// fileInputStream = new ByteArrayInputStream(xmlBytes);
				//
				// fileInputStream.read(xmlBytes, 0, xmlBytes.length);
				//
				// if (Utilidades.notNullAndNotEmpty(fileInputStream)) {
				// fileInputStream.close();
				// }
				//
				// zipStr = Base64.encodeBase64String(xmlBytes);

			} else {
				try {
					response.sendError(Constantes.HTTP_STATUS_EXPECTATION_FAILED,
							this.messageSource.getMessage("cuadro.validacion.zipNoDisponible", null, LocaleContextHolder.getLocale()) + "  ");
				} catch (IOException e) {
					Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al devolver el callback del error - "
							+ e);
				}
			}
		} catch (FileNotFoundException e) {
			Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al obtener el XML - " + e);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			Ab18aMonitorController.logger.error("Ab18aMonitorController - descargarEnviado() - Error al escribir el XML en el fichero - " + e);
		}
		// return zipStr;
	}

	/**
	 * @param messageSource
	 *            ReloadableResourceBundleMessageSource-.
	 */
	public void setMessageSource(ReloadableResourceBundleMessageSource messageSource) {
		this.messageSource = messageSource;
	}

	private Ab18aPeticionDetalle parsearFechas(Ab18aPeticionDetalle detalle) throws ParseException {
		String idioma = Utilidades.getIdiomaActual();
		if (idioma.equals("es")) {
			if (Utilidades.notNullAndNotEmpty(detalle.getFechaAlta())) {
				detalle.setFechaAltaFormat(parsearFecha(detalle.getFechaAlta(), "es"));
			}
			if (Utilidades.notNullAndNotEmpty(detalle.getFechaProgEnvio())) {
				detalle.setFechaProgEnvioFormat(parsearFecha(detalle.getFechaProgEnvio(), "es"));
			}

			if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta())) {
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaActualizacion())) {
					detalle.setFechaActualizacionFormat(parsearFecha(detalle.getDatosRespuesta().getFechaActualizacion(), "es"));
				}
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaEnvioBoletin())) {
					detalle.setFechaEnvioBoletinFormat(parsearFecha(detalle.getDatosRespuesta().getFechaEnvioBoletin(), "es"));
				}
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaPublicacion())) {
					detalle.setFechaPublicacionFormat(parsearFecha(detalle.getDatosRespuesta().getFechaPublicacion(), "es"));
				}
			}
		} else if (idioma.equals("eu")) {
			if (Utilidades.notNullAndNotEmpty(detalle.getFechaAlta())) {
				detalle.setFechaAltaFormat(parsearFecha(detalle.getFechaAlta(), "eu"));
			}
			if (Utilidades.notNullAndNotEmpty(detalle.getFechaProgEnvio())) {
				detalle.setFechaProgEnvioFormat(parsearFecha(detalle.getFechaProgEnvio(), "eu"));
			}
			if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta())) {
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaActualizacion())) {
					detalle.setFechaActualizacionFormat(parsearFecha(detalle.getDatosRespuesta().getFechaActualizacion(), "eu"));
				}
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaEnvioBoletin())) {
					detalle.setFechaEnvioBoletinFormat(parsearFecha(detalle.getDatosRespuesta().getFechaEnvioBoletin(), "eu"));
				}
				if (Utilidades.notNullAndNotEmpty(detalle.getDatosRespuesta().getFechaPublicacion())) {
					detalle.setFechaPublicacionFormat(parsearFecha(detalle.getDatosRespuesta().getFechaPublicacion(), "eu"));
				}
			}
		}

		return detalle;
	}

	private String parsearFecha(Date fecha, String idioma) throws ParseException {
		SimpleDateFormat sdf = null;
		if (idioma.equals("es")) {
			sdf = new SimpleDateFormat("dd/MM/yyyy");
		} else {
			sdf = new SimpleDateFormat("yyyy/MM/dd");
		}

		String dateAsString = sdf.format(fecha);

		return dateAsString;
	}
}
